'use client';

import { useState, useEffect } from 'react';
import { 
  Plus,
  Loader2,
  Edit,
  Trash2,
  Save,
  X,
  Eye,
  EyeOff,
  HelpCircle,
  ExternalLink,
  GripVertical,
  Download,
  CheckCircle
} from 'lucide-react';
import Link from 'next/link';

interface Faq {
  id: string;
  question: string;
  answer: string;
  category: string | null;
  sort_order: number;
  is_active: boolean;
  created_at: string;
  updated_at: string;
}

const categories = ['运输问题', '清关问题', '费用问题', '合作问题', '其他'];

// 默认常见问题数据
const defaultFaqs = [
  {
    question: '从中国发货到俄罗斯需要多长时间？',
    answer: '根据运输方式不同，时效也有所差异：\n\n• 铁路运输：约15-25天\n• 公路运输：约12-20天\n• 空运：约3-7天\n\n具体时效会根据起运城市、目的地城市、清关情况等因素有所调整，建议您提交询价单获取准确时效。',
    category: '运输问题',
    sortOrder: 1
  },
  {
    question: '货物清关需要哪些文件？',
    answer: '中俄跨境运输通常需要以下清关文件：\n\n• 贸易合同\n• 商业发票\n• 装箱单\n• 原产地证明\n• 运输合同\n• 危险品需提供相关证明（如有）\n\n我们的专业团队会协助您准备和审核所有必要文件。',
    category: '清关问题',
    sortOrder: 2
  },
  {
    question: '运费是如何计算的？',
    answer: '运费计算主要考虑以下因素：\n\n• 货物重量和体积（取较大值计费）\n• 运输方式（铁路/公路/空运）\n• 起运地和目的地\n• 货物类型（普货/危险品/冷链等）\n• 是否需要上门提货和送货上门\n\n您可以提交询价单，我们会在24小时内为您提供详细报价。',
    category: '费用问题',
    sortOrder: 3
  },
  {
    question: '是否提供货物保险服务？',
    answer: '是的，我们提供全程货物保险服务。\n\n• 保险费率：货值的0.3%-0.5%\n• 保险范围：全程运输（包括装卸、运输、清关）\n• 理赔时效：确认损失后7个工作日内\n\n建议贵重货物购买保险，保障您的利益。',
    category: '费用问题',
    sortOrder: 4
  },
  {
    question: '如何查询货物运输状态？',
    answer: '我们提供多种查询方式：\n\n1. 在线查询：登录用户中心，查看订单详情\n2. 实时沟通：通过订单页面与我们在线沟通\n3. 客服热线：拨打客服电话咨询\n\n运输过程中，我们会实时更新货物状态，让您随时掌握货物动态。',
    category: '运输问题',
    sortOrder: 5
  },
  {
    question: '可以运输哪些类型的货物？',
    answer: '我们承运的货物类型包括：\n\n• 普通货物：服装、电子产品、日用品等\n• 大件货物：机械设备、工程材料等\n• 危险品：化工产品、易燃物品（需特殊申报）\n• 冷链货物：食品、药品等温控货物\n\n部分特殊货物需要额外审批，详情请咨询客服。',
    category: '运输问题',
    sortOrder: 6
  },
  {
    question: '俄罗斯哪些城市可以送达？',
    answer: '我们覆盖俄罗斯主要城市和地区：\n\n• 莫斯科、圣彼得堡\n• 叶卡捷琳堡、新西伯利亚\n• 喀山、下诺夫哥罗德\n• 海参崴、哈巴罗夫斯克\n• 其他城市可根据需求安排\n\n基本上可以覆盖俄罗斯全境，偏远地区可能需要额外时间。',
    category: '运输问题',
    sortOrder: 7
  },
  {
    question: '货物损坏或丢失如何处理？',
    answer: '如发生货物损坏或丢失，处理流程如下：\n\n1. 第一时间联系客服登记\n2. 提供货物照片和相关证明材料\n3. 我们会立即启动调查程序\n4. 如已投保，按保险流程理赔\n5. 未投保货物按运输合同条款处理\n\n我们会全力协助处理，确保您的权益。',
    category: '运输问题',
    sortOrder: 8
  },
  {
    question: '支持哪些支付方式？',
    answer: '我们支持多种支付方式：\n\n• 银行转账（人民币/美元/卢布）\n• 支付宝、微信支付\n• 信用证付款\n\n一般流程：签约后支付30%定金，货物到达目的地后支付尾款。具体可协商。',
    category: '费用问题',
    sortOrder: 9
  },
  {
    question: '如何成为长期合作客户？',
    answer: '我们欢迎长期合作，合作客户可享受：\n\n• 优惠的运费价格\n• 优先安排运输\n• 专属客服对接\n• 月结账期\n\n如有意向，请联系我们的商务部门，我们会为您制定专属合作方案。',
    category: '合作问题',
    sortOrder: 10
  },
  {
    question: '清关一般需要多长时间？',
    answer: '清关时间因货物类型和口岸不同有所差异：\n\n• 普通货物：1-3个工作日\n• 需商检货物：3-5个工作日\n• 特殊货物：5-7个工作日\n\n我们有专业的清关团队，熟悉各项流程，能够确保货物快速通关。',
    category: '清关问题',
    sortOrder: 11
  },
  {
    question: '出口俄罗斯需要注意什么？',
    answer: '出口俄罗斯需要注意以下几点：\n\n• 确保货物符合俄罗斯进口标准\n• 准备完整的清关文件\n• 了解俄罗斯进口关税政策\n• 注意俄罗斯进口限制清单\n\n我们的专业团队会全程指导，确保出口顺利。',
    category: '清关问题',
    sortOrder: 12
  },
  {
    question: '可以提供上门提货服务吗？',
    answer: '是的，我们提供中国全境上门提货服务：\n\n• 覆盖主要城市：北京、上海、广州、深圳等\n• 提货费用根据距离和货物量计算\n• 需提前预约，建议至少提前2天\n\n提货服务与运输服务无缝衔接，一站式解决您的物流需求。',
    category: '运输问题',
    sortOrder: 13
  },
  {
    question: '货物包装有什么要求？',
    answer: '为确保运输安全，货物包装需满足：\n\n• 外包装坚固，能承受堆码和运输颠簸\n• 易碎品需加装缓冲材料\n• 防水防潮处理\n• 每件货物粘贴清晰标签\n\n如需包装服务，我们可以提供专业的包装解决方案。',
    category: '运输问题',
    sortOrder: 14
  },
  {
    question: '如何投诉或建议？',
    answer: '我们重视每一位客户的反馈：\n\n• 在线反馈：通过订单系统提交\n• 客服热线：工作时间随时拨打\n• 邮箱：support@xinqi-freight.com\n\n我们承诺在24小时内回复您的投诉或建议，持续改进服务质量。',
    category: '合作问题',
    sortOrder: 15
  }
];

export default function AdminFaqsPage() {
  const [faqs, setFaqs] = useState<Faq[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [editingFaq, setEditingFaq] = useState<Faq | null>(null);
  const [formData, setFormData] = useState({
    question: '',
    answer: '',
    category: '',
    sort_order: 0,
    is_active: true
  });
  const [saving, setSaving] = useState(false);
  const [importing, setImporting] = useState(false);
  const [initializing, setInitializing] = useState(false);
  const [tableExists, setTableExists] = useState<boolean | null>(null);

  // 获取常见问题列表
  const fetchFaqs = async () => {
    setIsLoading(true);
    try {
      const response = await fetch('/api/admin/faqs');
      const data = await response.json();
      if (response.ok) {
        setFaqs(data.faqs || []);
        setTableExists(true);
      } else if (data.code === 'PGRST205' || data.code === '42P01') {
        // 表不存在
        setTableExists(false);
        setFaqs([]);
      }
    } catch (error) {
      console.error('Error fetching faqs:', error);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchFaqs();
  }, []);

  // 初始化数据库表和默认数据
  const handleInitDatabase = async () => {
    if (!confirm('确定要初始化数据库吗？\n\n这将创建必要的表并导入15个默认常见问题。')) return;

    setInitializing(true);
    try {
      const response = await fetch('/api/admin/init-content', {
        method: 'POST'
      });
      const data = await response.json();
      
      if (data.success) {
        alert('初始化成功！');
        fetchFaqs();
      } else if (data.sqlScript) {
        // 表不存在，需要手动执行SQL
        alert('数据库表不存在，请在Supabase控制台执行以下SQL脚本：\n\nsql/init-cases-faqs.sql\n\n执行完成后刷新此页面。');
      } else {
        alert(data.error || '初始化失败');
      }
    } catch (error) {
      console.error('Error initializing:', error);
      alert('初始化失败');
    } finally {
      setInitializing(false);
    }
  };

  // 打开新增模态框
  const openAddModal = () => {
    setEditingFaq(null);
    setFormData({
      question: '',
      answer: '',
      category: '',
      sort_order: 0,
      is_active: true
    });
    setShowModal(true);
  };

  // 打开编辑模态框
  const openEditModal = (faq: Faq) => {
    setEditingFaq(faq);
    setFormData({
      question: faq.question,
      answer: faq.answer,
      category: faq.category || '',
      sort_order: faq.sort_order || 0,
      is_active: faq.is_active
    });
    setShowModal(true);
  };

  // 关闭模态框
  const closeModal = () => {
    setShowModal(false);
    setEditingFaq(null);
  };

  // 保存常见问题
  const handleSave = async () => {
    if (!formData.question.trim()) {
      alert('请输入问题');
      return;
    }
    if (!formData.answer.trim()) {
      alert('请输入答案');
      return;
    }

    setSaving(true);
    try {
      const url = editingFaq 
        ? `/api/admin/faqs/${editingFaq.id}`
        : '/api/admin/faqs';
      const method = editingFaq ? 'PUT' : 'POST';

      // 转换字段名为 camelCase
      const payload = {
        question: formData.question,
        answer: formData.answer,
        category: formData.category || undefined,
        sortOrder: formData.sort_order,
        isActive: formData.is_active
      };

      const response = await fetch(url, {
        method,
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(payload)
      });

      const data = await response.json();
      if (response.ok) {
        closeModal();
        fetchFaqs();
      } else {
        alert(data.error || '保存失败');
      }
    } catch (error) {
      console.error('Error saving faq:', error);
      alert('保存失败');
    } finally {
      setSaving(false);
    }
  };

  // 删除常见问题
  const handleDelete = async (id: string) => {
    if (!confirm('确定要删除这个问题吗？')) return;

    try {
      const response = await fetch(`/api/admin/faqs/${id}`, {
        method: 'DELETE'
      });

      if (response.ok) {
        fetchFaqs();
      } else {
        const data = await response.json();
        alert(data.error || '删除失败');
      }
    } catch (error) {
      console.error('Error deleting faq:', error);
      alert('删除失败');
    }
  };

  // 切换状态
  const toggleActive = async (faq: Faq) => {
    try {
      const response = await fetch(`/api/admin/faqs/${faq.id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          ...faq,
          isActive: !faq.is_active
        })
      });

      if (response.ok) {
        fetchFaqs();
      }
    } catch (error) {
      console.error('Error toggling faq:', error);
    }
  };

  // 导入默认问题
  const handleImportDefaults = async () => {
    if (!confirm('确定要导入默认的15个常见问题吗？\n\n注意：已存在的问题不会被重复导入。')) return;

    setImporting(true);
    try {
      const response = await fetch('/api/admin/faqs/batch', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ faqs: defaultFaqs })
      });

      const data = await response.json();
      if (response.ok) {
        alert(`成功导入 ${data.imported || 0} 个问题`);
        fetchFaqs();
      } else {
        alert(data.error || '导入失败');
      }
    } catch (error) {
      console.error('Error importing faqs:', error);
      alert('导入失败');
    } finally {
      setImporting(false);
    }
  };

  return (
    <div className="space-y-6">
      {/* 页面标题 */}
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">常见问题管理</h1>
          <p className="text-gray-500 mt-1">管理网站常见问题内容，共 {faqs.length} 个问题</p>
        </div>
        <div className="flex items-center gap-3">
          <Link 
            href="/faqs" 
            target="_blank"
            className="inline-flex items-center gap-2 px-4 py-2 text-gray-600 hover:text-blue-600 bg-gray-100 rounded-lg transition-colors text-sm"
          >
            <ExternalLink className="w-4 h-4" />
            预览页面
          </Link>
          <button
            onClick={openAddModal}
            className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium"
          >
            <Plus className="w-4 h-4" />
            新增问题
          </button>
        </div>
      </div>

      {/* 问题列表 */}
      {isLoading ? (
        <div className="flex items-center justify-center py-20">
          <Loader2 className="w-8 h-8 text-blue-600 animate-spin" />
        </div>
      ) : tableExists === false ? (
        <div className="bg-white rounded-2xl p-12 border border-gray-100 text-center">
          <HelpCircle className="w-12 h-12 text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">数据库表未初始化</h3>
          <p className="text-gray-500 mb-6">请点击下方按钮初始化数据库表和默认数据</p>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-3">
            <button
              onClick={handleInitDatabase}
              disabled={initializing}
              className="inline-flex items-center gap-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors font-medium disabled:opacity-50"
            >
              {initializing ? (
                <Loader2 className="w-4 h-4 animate-spin" />
              ) : (
                <CheckCircle className="w-4 h-4" />
              )}
              初始化数据库
            </button>
          </div>
          <p className="text-xs text-gray-400 mt-4">
            或者手动在Supabase控制台执行 <code className="bg-gray-100 px-1 rounded">sql/init-cases-faqs.sql</code>
          </p>
        </div>
      ) : faqs.length === 0 ? (
        <div className="bg-white rounded-2xl p-12 border border-gray-100 text-center">
          <HelpCircle className="w-12 h-12 text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">暂无常见问题</h3>
          <p className="text-gray-500 mb-6">点击"导入默认问题"快速添加15个预设问题，或手动新增问题</p>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-3">
            <button
              onClick={handleImportDefaults}
              disabled={importing}
              className="inline-flex items-center gap-2 px-4 py-2 text-green-600 bg-green-50 hover:bg-green-100 rounded-lg transition-colors text-sm font-medium disabled:opacity-50"
            >
              {importing ? (
                <Loader2 className="w-4 h-4 animate-spin" />
              ) : (
                <Download className="w-4 h-4" />
              )}
              导入默认问题
            </button>
            <button
              onClick={openAddModal}
              className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium"
            >
              <Plus className="w-4 h-4" />
              手动新增
            </button>
          </div>
        </div>
      ) : (
        <div className="bg-white rounded-2xl border border-gray-100 overflow-hidden">
          <div className="divide-y divide-gray-100">
            {faqs.map((faq) => (
              <div 
                key={faq.id}
                className={`p-5 hover:bg-gray-50/50 transition-colors ${
                  !faq.is_active ? 'opacity-60 bg-gray-50' : ''
                }`}
              >
                <div className="flex items-start gap-4">
                  {/* 排序号 */}
                  <div className="flex-shrink-0 w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center">
                    <span className="text-sm font-medium text-gray-600">{faq.sort_order}</span>
                  </div>

                  {/* 内容 */}
                  <div className="flex-1 min-w-0">
                    <div className="flex items-start gap-3">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          <h3 className="font-medium text-gray-900">{faq.question}</h3>
                          {faq.category && (
                            <span className="text-xs px-2 py-0.5 bg-blue-50 text-blue-600 rounded-full whitespace-nowrap">
                              {faq.category}
                            </span>
                          )}
                          {!faq.is_active && (
                            <span className="text-xs px-2 py-0.5 bg-gray-100 text-gray-500 rounded-full">
                              已隐藏
                            </span>
                          )}
                        </div>
                        <p className="text-sm text-gray-500 line-clamp-2">{faq.answer}</p>
                      </div>

                      {/* 操作按钮 */}
                      <div className="flex items-center gap-1">
                        <button
                          onClick={() => toggleActive(faq)}
                          className={`p-2 rounded-lg transition-colors ${
                            faq.is_active
                              ? 'text-green-600 hover:bg-green-50'
                              : 'text-gray-400 hover:bg-gray-100'
                          }`}
                          title={faq.is_active ? '点击隐藏' : '点击显示'}
                        >
                          {faq.is_active ? <Eye className="w-4 h-4" /> : <EyeOff className="w-4 h-4" />}
                        </button>
                        <button
                          onClick={() => openEditModal(faq)}
                          className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                          title="编辑"
                        >
                          <Edit className="w-4 h-4" />
                        </button>
                        <button
                          onClick={() => handleDelete(faq.id)}
                          className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition-colors"
                          title="删除"
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* 编辑模态框 */}
      {showModal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/50">
          <div className="bg-white rounded-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            {/* 头部 */}
            <div className="flex items-center justify-between p-5 border-b border-gray-100">
              <h2 className="text-lg font-semibold text-gray-900">
                {editingFaq ? '编辑问题' : '新增问题'}
              </h2>
              <button onClick={closeModal} className="p-2 hover:bg-gray-100 rounded-lg">
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>

            {/* 表单内容 */}
            <div className="p-5 space-y-5">
              {/* 问题 */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1.5">
                  问题 <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  value={formData.question}
                  onChange={(e) => setFormData(prev => ({ ...prev, question: e.target.value }))}
                  className="w-full px-4 py-2.5 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                  placeholder="请输入问题"
                />
              </div>

              {/* 答案 */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1.5">
                  答案 <span className="text-red-500">*</span>
                </label>
                <textarea
                  value={formData.answer}
                  onChange={(e) => setFormData(prev => ({ ...prev, answer: e.target.value }))}
                  className="w-full px-4 py-2.5 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                  rows={6}
                  placeholder="请输入答案（支持换行）"
                />
              </div>

              {/* 分类和排序 */}
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1.5">分类</label>
                  <select
                    value={formData.category}
                    onChange={(e) => setFormData(prev => ({ ...prev, category: e.target.value }))}
                    className="w-full px-4 py-2.5 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    <option value="">选择分类</option>
                    {categories.map(cat => (
                      <option key={cat} value={cat}>{cat}</option>
                    ))}
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1.5">排序</label>
                  <input
                    type="number"
                    value={formData.sort_order}
                    onChange={(e) => setFormData(prev => ({ ...prev, sort_order: parseInt(e.target.value) || 0 }))}
                    className="w-full px-4 py-2.5 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                  />
                </div>
              </div>

              {/* 状态 */}
              <div className="flex items-center gap-3">
                <input
                  type="checkbox"
                  id="faq_is_active"
                  checked={formData.is_active}
                  onChange={(e) => setFormData(prev => ({ ...prev, is_active: e.target.checked }))}
                  className="w-4 h-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                />
                <label htmlFor="faq_is_active" className="text-sm text-gray-700">显示在前台页面</label>
              </div>
            </div>

            {/* 底部按钮 */}
            <div className="flex items-center justify-end gap-3 p-5 border-t border-gray-100">
              <button
                onClick={closeModal}
                className="px-4 py-2 text-gray-600 hover:bg-gray-100 rounded-lg transition-colors text-sm"
              >
                取消
              </button>
              <button
                onClick={handleSave}
                disabled={saving}
                className="inline-flex items-center gap-2 px-5 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium disabled:opacity-50"
              >
                {saving ? (
                  <>
                    <Loader2 className="w-4 h-4 animate-spin" />
                    保存中...
                  </>
                ) : (
                  <>
                    <Save className="w-4 h-4" />
                    保存
                  </>
                )}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
