import { NextRequest, NextResponse } from 'next/server';
import { getSupabaseClient } from '@/storage/database/supabase-client';

// 验证管理员身份
async function verifyAdmin(request: NextRequest): Promise<boolean> {
  const adminId = request.cookies.get('admin_id')?.value;
  if (!adminId) return false;
  
  try {
    const client = getSupabaseClient();
    const { data: admin } = await client
      .from('admins')
      .select('id')
      .eq('id', adminId)
      .eq('is_active', true)
      .single();
    
    return !!admin;
  } catch {
    return false;
  }
}

// GET: 获取所有多语言内容（管理员）
export async function GET(request: NextRequest) {
  try {
    if (!(await verifyAdmin(request))) {
      return NextResponse.json({ error: '未授权' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const group = searchParams.get('group');

    const client = getSupabaseClient();
    let query = client
      .from('i18n_content')
      .select('id, group_name, key_name, content_zh, content_ru, description, sort_order, created_at, updated_at');

    if (group) {
      query = query.eq('group_name', group);
    }

    const { data: content, error } = await query.order('sort_order', { ascending: true });

    if (error) {
      console.error('Error fetching i18n content:', error);
      return NextResponse.json({ error: '获取内容失败' }, { status: 500 });
    }

    // 转换字段名称以兼容前端
    const formattedContent = (content || []).map(item => ({
      id: item.id,
      groupName: item.group_name,
      keyName: item.key_name,
      contentZh: item.content_zh,
      contentRu: item.content_ru,
      description: item.description,
      sortOrder: item.sort_order,
      createdAt: item.created_at,
      updatedAt: item.updated_at,
    }));

    return NextResponse.json({ success: true, content: formattedContent });
  } catch (error) {
    console.error('Error fetching i18n content:', error);
    return NextResponse.json({ error: '获取内容失败' }, { status: 500 });
  }
}

// PUT: 批量更新多语言内容（管理员）
export async function PUT(request: NextRequest) {
  try {
    if (!(await verifyAdmin(request))) {
      return NextResponse.json({ error: '未授权' }, { status: 401 });
    }

    const body = await request.json();
    const items = Array.isArray(body) ? body : [body];

    const client = getSupabaseClient();
    
    // 批量更新
    for (const item of items) {
      const { error } = await client
        .from('i18n_content')
        .update({
          content_zh: item.contentZh,
          content_ru: item.contentRu,
          updated_at: new Date().toISOString(),
        })
        .eq('id', item.id);

      if (error) {
        console.error('Error updating i18n content:', error);
        return NextResponse.json({ error: '更新失败' }, { status: 500 });
      }
    }

    return NextResponse.json({ 
      success: true, 
      message: '更新成功',
      updated: items.length 
    });
  } catch (error) {
    console.error('Error updating i18n content:', error);
    return NextResponse.json({ error: '更新失败' }, { status: 500 });
  }
}

// POST: 初始化默认多语言内容（管理员）
export async function POST(request: NextRequest) {
  try {
    if (!(await verifyAdmin(request))) {
      return NextResponse.json({ error: '未授权' }, { status: 401 });
    }

    // 默认首页内容
    const defaultContent = [
      // Hero区域
      { id: 'hero_title', group_name: 'home', key_name: 'heroTitle', content_zh: '打通中俄物流全链路', content_ru: 'Полная логистическая цепочка Китай-Россия', description: '首页主标题', sort_order: 1 },
      { id: 'hero_subtitle', group_name: 'home', key_name: 'heroSubtitle', content_zh: '让跨境运输更简单', content_ru: 'Упрощаем трансграничные перевозки', description: '首页副标题', sort_order: 2 },
      { id: 'hero_desc', group_name: 'home', key_name: 'heroDesc', content_zh: '专业的中俄跨境物流解决方案提供商，为您的货物保驾护航', content_ru: 'Профессиональный провайдер логистических решений Китай-Россия', description: '首页描述', sort_order: 3 },
      
      // 数据统计
      { id: 'stats_years_value', group_name: 'home', key_name: 'statsYearsValue', content_zh: '10', content_ru: '10', description: '数据统计-年数', sort_order: 10 },
      { id: 'stats_years_label', group_name: 'home', key_name: 'statsYearsLabel', content_zh: '行业深耕年限', content_ru: 'лет опыта', description: '数据统计-年数标签', sort_order: 11 },
      { id: 'stats_clients_value', group_name: 'home', key_name: 'statsClientsValue', content_zh: '3000', content_ru: '3000', description: '数据统计-客户数', sort_order: 12 },
      { id: 'stats_clients_label', group_name: 'home', key_name: 'statsClientsLabel', content_zh: '服务企业数量', content_ru: 'клиентов', description: '数据统计-客户标签', sort_order: 13 },
      { id: 'stats_satisfaction_value', group_name: 'home', key_name: 'statsSatisfactionValue', content_zh: '99', content_ru: '99', description: '数据统计-满意度', sort_order: 14 },
      { id: 'stats_satisfaction_label', group_name: 'home', key_name: 'statsSatisfactionLabel', content_zh: '客户满意度', content_ru: 'удовлетворенность', description: '数据统计-满意度标签', sort_order: 15 },
      { id: 'stats_service_value', group_name: 'home', key_name: 'statsServiceValue', content_zh: '24', content_ru: '24', description: '数据统计-服务时长', sort_order: 16 },
      { id: 'stats_service_label', group_name: 'home', key_name: 'statsServiceLabel', content_zh: '全天候服务', content_ru: 'круглосуточно', description: '数据统计-服务标签', sort_order: 17 },

      // 关于我们
      { id: 'about_title', group_name: 'home', key_name: 'aboutTitle', content_zh: '深耕中俄货运领域', content_ru: 'Эксперты логистики', description: '关于我们标题', sort_order: 20 },
      { id: 'about_subtitle', group_name: 'home', key_name: 'aboutSubtitle', content_zh: '成为可靠的跨境桥梁', content_ru: 'Китай — Россия', description: '关于我们副标题', sort_order: 21 },
      { id: 'about_desc1', group_name: 'home', key_name: 'aboutDesc1', content_zh: '满洲里欣琪国际货运代理有限责任公司深耕中俄货运领域多年，凭借扎根满洲里口岸的地理优势与对中俄贸易规则的深刻理解，成为连接两国货物往来的可靠桥梁。', content_ru: 'Маньчжурия Синьци Международное Экспедирование — эксперт в сфере логистики Китай-Россия с многолетним опытом работы на границе.', description: '关于我们描述1', sort_order: 22 },
      { id: 'about_desc2', group_name: 'home', key_name: 'aboutDesc2', content_zh: '作为中俄贸易的见证者与参与者，欣琪国际始终以"安全、高效、透明"为服务准则，累计为数千家企业完成跨境货物运输。', content_ru: 'Мы обслужили тысячи компаний, обеспечивая безопасность, эффективность и прозрачность на каждом этапе.', description: '关于我们描述2', sort_order: 23 },

      // 服务内容
      { id: 'services_title', group_name: 'home', key_name: 'servicesTitle', content_zh: '全方位中俄货运解决方案', content_ru: 'Полный спектр услуг', description: '服务内容标题', sort_order: 30 },
      { id: 'services_desc', group_name: 'home', key_name: 'servicesDesc', content_zh: '满足不同客户的多样化需求，提供一站式跨境物流服务', content_ru: 'Индивидуальные решения для каждого клиента', description: '服务内容描述', sort_order: 31 },

      // 核心竞争力
      { id: 'advantages_title', group_name: 'home', key_name: 'advantagesTitle', content_zh: '三重专业保障', content_ru: 'Три уровня гарантии', description: '核心竞争力标题', sort_order: 40 },
      { id: 'advantages_desc', group_name: 'home', key_name: 'advantagesDesc', content_zh: '让我们成为中俄货运领域的可靠合作伙伴', content_ru: 'Почему нам доверяют', description: '核心竞争力描述', sort_order: 41 },

      // 案例展示
      { id: 'cases_title', group_name: 'home', key_name: 'casesTitle', content_zh: '值得信赖的合作伙伴', content_ru: 'Надежный партнер', description: '案例展示标题', sort_order: 50 },
      { id: 'cases_desc', group_name: 'home', key_name: 'casesDesc', content_zh: '我们已为数千家企业提供专业的中俄物流服务', content_ru: 'Мы обслужили тысячи клиентов', description: '案例展示描述', sort_order: 51 },

      // 在线询价
      { id: 'order_title', group_name: 'home', key_name: 'orderTitle', content_zh: '立即提交您的运输需求', content_ru: 'Отправьте заявку сейчас', description: '在线询价标题', sort_order: 60 },

      // 联系我们
      { id: 'contact_title', group_name: 'home', key_name: 'contactTitle', content_zh: '期待与您的合作', content_ru: 'Ждем вашего звонка', description: '联系我们标题', sort_order: 70 },

      // 客户类型
      { id: 'clients_title', group_name: 'home', key_name: 'clientsTitle', content_zh: '我们服务的客户类型', content_ru: 'Наши клиенты', description: '客户类型标题', sort_order: 80 },
    ];

    const client = getSupabaseClient();
    
    // 插入或更新
    let inserted = 0;
    let updated = 0;

    for (const item of defaultContent) {
      const { data: existing } = await client
        .from('i18n_content')
        .select('id')
        .eq('id', item.id)
        .limit(1);

      if (!existing || existing.length === 0) {
        const { error } = await client.from('i18n_content').insert({
          id: item.id,
          group_name: item.group_name,
          key_name: item.key_name,
          content_zh: item.content_zh,
          content_ru: item.content_ru,
          description: item.description,
          sort_order: item.sort_order,
          created_at: new Date().toISOString(),
          updated_at: new Date().toISOString(),
        });
        
        if (error) {
          console.error('Error inserting i18n content:', error);
        } else {
          inserted++;
        }
      } else {
        updated++;
      }
    }

    return NextResponse.json({ 
      success: true, 
      message: '初始化完成',
      inserted,
      updated,
      total: defaultContent.length
    });
  } catch (error) {
    console.error('Error initializing i18n content:', error);
    return NextResponse.json({ error: '初始化失败' }, { status: 500 });
  }
}
