import { NextResponse } from 'next/server';
import { getSupabaseClient } from '@/storage/database/supabase-client';
import { createHash } from 'crypto';

// 初始化管理员表和默认管理员账号
export async function POST() {
  try {
    const client = getSupabaseClient();
    
    // 检查是否已存在管理员
    const { data: existingAdmins } = await client
      .from('admins')
      .select('id')
      .limit(1);
    
    if (existingAdmins && existingAdmins.length > 0) {
      return NextResponse.json({ 
        message: '管理员表已存在',
        adminExists: true 
      });
    }
    
    // 创建默认管理员账号
    // 用户名: admin, 密码: admin123
    const defaultPassword = createHash('sha256').update('admin123').digest('hex');
    
    const { error } = await client
      .from('admins')
      .insert({
        username: 'admin',
        password: defaultPassword,
        name: '系统管理员',
        role: 'super_admin',
      });
    
    if (error) {
      // 如果表不存在，返回提示
      if (error.code === '42P01') {
        return NextResponse.json({ 
          error: '管理员表不存在，请先创建数据库表',
          hint: '请执行 CREATE TABLE admins ... 创建表',
          details: error.message 
        }, { status: 500 });
      }
      
      console.error('Error creating default admin:', error);
      return NextResponse.json({ 
        error: '创建默认管理员失败',
        details: error.message 
      }, { status: 500 });
    }
    
    return NextResponse.json({ 
      message: '管理员表初始化成功',
      defaultAdmin: {
        username: 'admin',
        password: 'admin123'
      }
    });
  } catch (error) {
    console.error('Error:', error);
    return NextResponse.json({ 
      error: '初始化失败',
      details: String(error)
    }, { status: 500 });
  }
}
