'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { 
  ChevronDown,
  ChevronUp,
  ArrowLeft,
  HelpCircle,
  Loader2,
  Search
} from 'lucide-react';
import { useLanguage } from '@/i18n/LanguageContext';
import { LanguageSwitcherSimple } from '@/components/language-switcher';

interface Faq {
  id: string;
  question: string;
  answer: string;
  category: string | null;
}

// 默认常见问题数据 - 中文
const defaultFaqsZh = [
  {
    question: '从中国发货到俄罗斯需要多长时间？',
    answer: '根据运输方式不同，时效也有所差异：\n\n• 铁路运输：约15-25天\n• 公路运输：约12-20天\n• 空运：约3-7天\n\n具体时效会根据起运城市、目的地城市、清关情况等因素有所调整，建议您提交询价单获取准确时效。',
    category: '运输问题'
  },
  {
    question: '货物清关需要哪些文件？',
    answer: '中俄跨境运输通常需要以下清关文件：\n\n• 贸易合同\n• 商业发票\n• 装箱单\n• 原产地证明\n• 运输合同\n• 危险品需提供相关证明（如有）\n\n我们的专业团队会协助您准备和审核所有必要文件。',
    category: '清关问题'
  },
  {
    question: '运费是如何计算的？',
    answer: '运费计算主要考虑以下因素：\n\n• 货物重量和体积（取较大值计费）\n• 运输方式（铁路/公路/空运）\n• 起运地和目的地\n• 货物类型（普货/危险品/冷链等）\n• 是否需要上门提货和送货上门\n\n您可以提交询价单，我们会在24小时内为您提供详细报价。',
    category: '费用问题'
  },
  {
    question: '是否提供货物保险服务？',
    answer: '是的，我们提供全程货物保险服务。\n\n• 保险费率：货值的0.3%-0.5%\n• 保险范围：全程运输（包括装卸、运输、清关）\n• 理赔时效：确认损失后7个工作日内\n\n建议贵重货物购买保险，保障您的利益。',
    category: '费用问题'
  },
  {
    question: '如何查询货物运输状态？',
    answer: '我们提供多种查询方式：\n\n1. 在线查询：登录用户中心，查看订单详情\n2. 实时沟通：通过订单页面与我们在线沟通\n3. 客服热线：拨打客服电话咨询\n\n运输过程中，我们会实时更新货物状态，让您随时掌握货物动态。',
    category: '运输问题'
  },
  {
    question: '可以运输哪些类型的货物？',
    answer: '我们承运的货物类型包括：\n\n• 普通货物：服装、电子产品、日用品等\n• 大件货物：机械设备、工程材料等\n• 危险品：化工产品、易燃物品（需特殊申报）\n• 冷链货物：食品、药品等温控货物\n\n部分特殊货物需要额外审批，详情请咨询客服。',
    category: '运输问题'
  },
  {
    question: '俄罗斯哪些城市可以送达？',
    answer: '我们覆盖俄罗斯主要城市和地区：\n\n• 莫斯科、圣彼得堡\n• 叶卡捷琳堡、新西伯利亚\n• 喀山、下诺夫哥罗德\n• 海参崴、哈巴罗夫斯克\n• 其他城市可根据需求安排\n\n基本上可以覆盖俄罗斯全境，偏远地区可能需要额外时间。',
    category: '运输问题'
  },
  {
    question: '货物损坏或丢失如何处理？',
    answer: '如发生货物损坏或丢失，处理流程如下：\n\n1. 第一时间联系客服登记\n2. 提供货物照片和相关证明材料\n3. 我们会立即启动调查程序\n4. 如已投保，按保险流程理赔\n5. 未投保货物按运输合同条款处理\n\n我们会全力协助处理，确保您的权益。',
    category: '运输问题'
  },
  {
    question: '支持哪些支付方式？',
    answer: '我们支持多种支付方式：\n\n• 银行转账（人民币/美元/卢布）\n• 支付宝、微信支付\n• 信用证付款\n\n一般流程：签约后支付30%定金，货物到达目的地后支付尾款。具体可协商。',
    category: '费用问题'
  },
  {
    question: '如何成为长期合作客户？',
    answer: '我们欢迎长期合作，合作客户可享受：\n\n• 优惠的运费价格\n• 优先安排运输\n• 专属客服对接\n• 月结账期\n\n如有意向，请联系我们的商务部门，我们会为您制定专属合作方案。',
    category: '合作问题'
  },
  {
    question: '清关一般需要多长时间？',
    answer: '清关时间因货物类型和口岸不同有所差异：\n\n• 普通货物：1-3个工作日\n• 需商检货物：3-5个工作日\n• 特殊货物：5-7个工作日\n\n我们有专业的清关团队，熟悉各项流程，能够确保货物快速通关。',
    category: '清关问题'
  },
  {
    question: '出口俄罗斯需要注意什么？',
    answer: '出口俄罗斯需要注意以下几点：\n\n• 确保货物符合俄罗斯进口标准\n• 准备完整的清关文件\n• 了解俄罗斯进口关税政策\n• 注意俄罗斯进口限制清单\n\n我们的专业团队会全程指导，确保出口顺利。',
    category: '清关问题'
  },
  {
    question: '可以提供上门提货服务吗？',
    answer: '是的，我们提供中国全境上门提货服务：\n\n• 覆盖主要城市：北京、上海、广州、深圳等\n• 提货费用根据距离和货物量计算\n• 需提前预约，建议至少提前2天\n\n提货服务与运输服务无缝衔接，一站式解决您的物流需求。',
    category: '运输问题'
  },
  {
    question: '货物包装有什么要求？',
    answer: '为确保运输安全，货物包装需满足：\n\n• 外包装坚固，能承受堆码和运输颠簸\n• 易碎品需加装缓冲材料\n• 防水防潮处理\n• 每件货物粘贴清晰标签\n\n如需包装服务，我们可以提供专业的包装解决方案。',
    category: '运输问题'
  },
  {
    question: '如何投诉或建议？',
    answer: '我们重视每一位客户的反馈：\n\n• 在线反馈：通过订单系统提交\n• 客服热线：工作时间随时拨打\n• 邮箱：mzlxqgj@163.com\n\n我们承诺在24小时内回复您的投诉或建议，持续改进服务质量。',
    category: '合作问题'
  }
];

// 默认常见问题数据 - 俄语
const defaultFaqsRu = [
  {
    question: 'Сколько времени занимает доставка из Китая в Россию?',
    answer: 'Время доставки зависит от способа транспортировки:\n\n• Ж/Д перевозки: около 15-25 дней\n• Автоперевозки: около 12-20 дней\n• Авиадоставка: около 3-7 дней\n\nТочное время зависит от города отправления, назначения и таможенного оформления.',
    category: 'Перевозки'
  },
  {
    question: 'Какие документы нужны для таможенного оформления?',
    answer: 'Для трансграничных перевозок Китай-Россия обычно требуются:\n\n• Внешнеторговый контракт\n• Коммерческий инвойс\n• Упаковочный лист\n• Сертификат происхождения\n• Транспортный договор\n• Для опасных грузов - соответствующие сертификаты\n\nНаша команда поможет подготовить все необходимые документы.',
    category: 'Таможня'
  },
  {
    question: 'Как рассчитывается стоимость перевозки?',
    answer: 'Стоимость зависит от следующих факторов:\n\n• Вес и объем груза (берется большее значение)\n• Способ перевозки (ж/д/авто/авиа)\n• Пункт отправления и назначения\n• Тип груза (обычный/опасный/холодильный)\n• Необходимость забора и доставки\n\nОтправьте заявку, и мы предоставим точный расчет в течение 24 часов.',
    category: 'Стоимость'
  },
  {
    question: 'Предоставляете ли вы страхование грузов?',
    answer: 'Да, мы предлагаем полное страхование грузов:\n\n• Тариф: 0.3%-0.5% от стоимости груза\n• Покрытие: весь маршрут (погрузка, перевозка, таможня)\n• Выплата: в течение 7 рабочих дней\n\nРекомендуем страховать ценные грузы.',
    category: 'Стоимость'
  },
  {
    question: 'Как отследить статус перевозки?',
    answer: 'Мы предлагаем несколько способов отслеживания:\n\n1. Онлайн: в личном кабинете\n2. Чат: общение с менеджером на странице заказа\n3. Телефон: звонок в службу поддержки\n\nМы обновляем статус в реальном времени.',
    category: 'Перевозки'
  },
  {
    question: 'Какие типы грузов вы перевозите?',
    answer: 'Мы перевозим:\n\n• Обычные грузы: одежда, электроника, товары народного потребления\n• Крупногабаритные: оборудование, стройматериалы\n• Опасные грузы: химия, легковоспламеняющиеся (требуется спецразрешение)\n• Рефрижераторные: продукты, лекарства\n\nДля некоторых грузов требуется дополнительное оформление.',
    category: 'Перевозки'
  },
  {
    question: 'В какие города России осуществляется доставка?',
    answer: 'Мы охватываем основные города России:\n\n• Москва, Санкт-Петербург\n• Екатеринбург, Новосибирск\n• Казань, Нижний Новгород\n• Владивосток, Хабаровск\n• Другие города по запросу\n\nПрактически вся территория России доступна для доставки.',
    category: 'Перевозки'
  },
  {
    question: 'Что делать при повреждении или потере груза?',
    answer: 'При повреждении или потере груза:\n\n1. Немедленно свяжитесь с нами\n2. Предоставьте фото и документы\n3. Мы запустим расследование\n4. При наличии страховки - выплата по полису\n5. Без страховки - по условиям договора\n\nМы сделаем всё для защиты ваших интересов.',
    category: 'Перевозки'
  },
  {
    question: 'Какие способы оплаты принимаются?',
    answer: 'Мы принимаем:\n\n• Банковский перевод (CNY/USD/RUB)\n• Alipay, WeChat Pay\n• Аккредитив\n\nОбычно: 30% предоплата, остаток по прибытии. Условия обсуждаемы.',
    category: 'Стоимость'
  },
  {
    question: 'Как стать постоянным клиентом?',
    answer: 'Постоянные клиенты получают:\n\n• Скидки на перевозки\n• Приоритетное обслуживание\n• Персонального менеджера\n• Отсрочку платежа\n\nСвяжитесь с нами для обсуждения условий сотрудничества.',
    category: 'Сотрудничество'
  },
  {
    question: 'Сколько времени занимает таможенное оформление?',
    answer: 'Время зависит от типа груза:\n\n• Обычные грузы: 1-3 рабочих дня\n• Грузы с сертификацией: 3-5 дней\n• Специальные грузы: 5-7 дней\n\nНаша команда экспертов обеспечивает быстрое оформление.',
    category: 'Таможня'
  },
  {
    question: 'На что обратить внимание при экспорте в Россию?',
    answer: 'При экспорте в Россию важно:\n\n• Соответствие российским стандартам\n• Полный пакет документов\n• Знание таможенных тарифов\n• Проверка на запреты и ограничения\n\nМы поможем на всех этапах.',
    category: 'Таможня'
  },
  {
    question: 'Можно ли заказать забор груза?',
    answer: 'Да, мы предоставляем забор груза по всему Китаю:\n\n• Основные города: Пекин, Шанхай, Гуанчжоу, Шэньчжэнь\n• Стоимость зависит от расстояния и объема\n• Рекомендуется бронировать за 2 дня\n\nЗабор и доставка интегрированы в наш сервис.',
    category: 'Перевозки'
  },
  {
    question: 'Какие требования к упаковке груза?',
    answer: 'Для безопасной перевозки упаковка должна:\n\n• Быть прочной, выдерживать штабелирование\n• Иметь амортизацию для хрупких грузов\n• Быть водонепроницаемой\n• Иметь четкие этикетки\n\nМы можем предоставить услуги по упаковке.',
    category: 'Перевозки'
  },
  {
    question: 'Как оставить жалобу или предложение?',
    answer: 'Мы ценим обратную связь:\n\n• Онлайн: через систему заказов\n• Телефон: в рабочее время\n• Email: mzlxqgj@163.com\n\nМы отвечаем в течение 24 часов.',
    category: 'Сотрудничество'
  }
];

export default function FaqsPage() {
  const { t, locale } = useLanguage();
  const [faqs, setFaqs] = useState<Faq[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [expandedId, setExpandedId] = useState<string | null>(null);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('all');

  useEffect(() => {
    const fetchFaqs = async () => {
      try {
        const response = await fetch('/api/faqs');
        const data = await response.json();
        if (response.ok) {
          if (data.faqs && data.faqs.length > 0) {
            setFaqs(data.faqs);
          } else {
            // 如果没有数据，使用默认数据
            const defaultFaqs = locale === 'zh' ? defaultFaqsZh : defaultFaqsRu;
            setFaqs(defaultFaqs.map((faq, index) => ({
              id: `default-${index}`,
              ...faq
            })));
          }
        }
      } catch (error) {
        console.error('Error fetching faqs:', error);
        // 出错时使用默认数据
        const defaultFaqs = locale === 'zh' ? defaultFaqsZh : defaultFaqsRu;
        setFaqs(defaultFaqs.map((faq, index) => ({
          id: `default-${index}`,
          ...faq
        })));
      } finally {
        setIsLoading(false);
      }
    };

    fetchFaqs();
  }, [locale]);

  // 获取所有分类
  const categories = ['all', ...new Set(faqs.map(faq => faq.category).filter((c): c is string => Boolean(c)))];

  // 过滤FAQ
  const filteredFaqs = faqs.filter(faq => {
    const matchesSearch = faq.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
                          faq.answer.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || faq.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  return (
    <div className="min-h-screen bg-gradient-to-b from-gray-50 to-white pt-20">
      {/* 导航栏 */}
      <nav className="fixed top-0 left-0 right-0 z-40 bg-white/80 backdrop-blur-lg border-b border-gray-100">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center gap-3">
              <Link href="/" className="flex items-center gap-3">
                <div className="w-11 h-11 bg-white rounded-xl flex items-center justify-center shadow-lg border border-gray-100">
                  <img src="/images/logo.png" alt={t.common.logo} className="w-9 h-9 object-contain" />
                </div>
                <div>
                  <h1 className="text-lg font-bold bg-gradient-to-r from-blue-600 to-indigo-700 bg-clip-text text-transparent">{t.common.logo}</h1>
                  <p className="text-xs text-gray-400 font-medium tracking-wide">{t.common.logoSub}</p>
                </div>
              </Link>
            </div>
            <div className="flex items-center gap-3">
              <LanguageSwitcherSimple />
              <Link 
                href="/" 
                className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:text-blue-600 transition-colors text-sm"
              >
                <ArrowLeft className="w-4 h-4" />
                {t.nav.home}
              </Link>
            </div>
          </div>
        </div>
      </nav>

      {/* 页面内容 */}
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        {/* 页面标题 */}
        <div className="text-center mb-10">
          <div className="inline-flex items-center justify-center w-16 h-16 bg-blue-100 rounded-full mb-4">
            <HelpCircle className="w-8 h-8 text-blue-600" />
          </div>
          <h1 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">{t.faqs.title}</h1>
          <p className="text-gray-500 max-w-2xl mx-auto">
            {t.faqs.subtitle}
          </p>
        </div>

        {/* 搜索和筛选 */}
        <div className="bg-white rounded-2xl border border-gray-100 p-4 mb-8 shadow-sm">
          <div className="flex flex-col sm:flex-row gap-4">
            {/* 搜索框 */}
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
              <input
                type="text"
                placeholder={t.faqs.searchPlaceholder}
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pl-10 pr-4 py-2.5 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
            
            {/* 分类筛选 */}
            <div className="flex gap-2 overflow-x-auto pb-1">
              {categories.map((cat) => (
                <button
                  key={cat}
                  onClick={() => setSelectedCategory(cat)}
                  className={`px-4 py-2 rounded-lg text-sm font-medium whitespace-nowrap transition-all ${
                    selectedCategory === cat
                      ? 'bg-blue-600 text-white'
                      : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                  }`}
                >
                  {cat === 'all' ? t.faqs.all : cat}
                </button>
              ))}
            </div>
          </div>
        </div>

        {/* FAQ列表 */}
        {isLoading ? (
          <div className="flex items-center justify-center py-20">
            <Loader2 className="w-8 h-8 text-blue-600 animate-spin" />
          </div>
        ) : filteredFaqs.length === 0 ? (
          <div className="text-center py-20 text-gray-500">
            {t.faqs.noResults}
          </div>
        ) : (
          <div className="space-y-3">
            {filteredFaqs.map((faq) => {
              const isExpanded = expandedId === faq.id;
              
              return (
                <div 
                  key={faq.id}
                  className="bg-white rounded-xl border border-gray-100 shadow-sm overflow-hidden"
                >
                  <button
                    onClick={() => setExpandedId(isExpanded ? null : faq.id)}
                    className="w-full flex items-start gap-3 p-5 text-left hover:bg-gray-50 transition-colors"
                  >
                    <div className={`w-8 h-8 rounded-lg flex items-center justify-center flex-shrink-0 transition-colors ${
                      isExpanded 
                        ? 'bg-blue-100 text-blue-600' 
                        : 'bg-gray-100 text-gray-400'
                    }`}>
                      {isExpanded ? (
                        <ChevronUp className="w-5 h-5" />
                      ) : (
                        <ChevronDown className="w-5 h-5" />
                      )}
                    </div>
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <span className="font-medium text-gray-900">{faq.question}</span>
                        {faq.category && (
                          <span className="text-xs px-2 py-0.5 bg-blue-50 text-blue-600 rounded-full">
                            {faq.category}
                          </span>
                        )}
                      </div>
                    </div>
                  </button>
                  
                  {isExpanded && (
                    <div className="px-5 pb-5 ml-12">
                      <div className="text-gray-600 text-sm leading-relaxed whitespace-pre-line">
                        {faq.answer}
                      </div>
                    </div>
                  )}
                </div>
              );
            })}
          </div>
        )}

        {/* 底部提示 */}
        <div className="mt-12 text-center">
          <p className="text-gray-500 mb-4">{t.faqs.notFound}</p>
          <Link 
            href="/#contact" 
            className="inline-flex items-center gap-2 px-6 py-3 bg-blue-600 text-white font-medium rounded-xl hover:bg-blue-700 transition-colors"
          >
            {t.faqs.contactUs}
          </Link>
        </div>
      </div>
    </div>
  );
}
