'use client';

import { useState, useEffect, useMemo } from 'react';
import Link from 'next/link';
import { 
  Truck, 
  Ship, 
  Package, 
  Globe, 
  Users, 
  Award, 
  CheckCircle2, 
  Phone, 
  Mail, 
  MapPin,
  ArrowRight,
  Train,
  FileText,
  Building2,
  Wrench,
  ArrowLeftRight,
  Sparkles,
  Route,
  Car,
  Shield,
  Clock,
  Target,
  ChevronDown,
  Send,
  MessageSquare,
  User,
  ClipboardList,
  Loader2
} from 'lucide-react';
import { useLanguage } from '@/i18n/LanguageContext';
import { LanguageSwitcherSimple } from '@/components/language-switcher';

// 动画数字组件
function AnimatedNumber({ end, suffix = '' }: { end: number; suffix?: string }) {
  const [count, setCount] = useState(0);
  
  useEffect(() => {
    let start = 0;
    const duration = 2000;
    const increment = end / (duration / 16);
    
    const timer = setInterval(() => {
      start += increment;
      if (start >= end) {
        setCount(end);
        clearInterval(timer);
      } else {
        setCount(Math.floor(start));
      }
    }, 16);
    
    return () => clearInterval(timer);
  }, [end]);
  
  return <span>{count}{suffix}</span>;
}

export default function Home() {
  const { t, locale, getDbContent, isLoadingDb } = useLanguage();
  
  // 辅助函数：获取数据库内容或默认值
  const getContent = (key: string, defaultZh: string, defaultRu: string): string => {
    const dbValue = getDbContent(key);
    if (dbValue) return dbValue;
    return locale === 'zh' ? defaultZh : defaultRu;
  };
  
  // 用户登录状态
  const [isLoggedIn, setIsLoggedIn] = useState(false);
  const [customerInfo, setCustomerInfo] = useState<{
    name: string;
    phone: string;
    email: string | null;
  } | null>(null);
  
  // 订单表单状态
  const [formData, setFormData] = useState({
    contactName: '',
    contactPhone: '',
    contactEmail: '',
    originCity: '',
    destinationCity: '',
    customsCity: '',
    notes: '',
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitStatus, setSubmitStatus] = useState<'idle' | 'success' | 'error'>('idle');
  const [errorMessage, setErrorMessage] = useState('');

  // 检查登录状态
  useEffect(() => {
    const checkAuth = async () => {
      try {
        const response = await fetch('/api/auth/me');
        const data = await response.json();
        if (data.authenticated && data.customer) {
          setIsLoggedIn(true);
          setCustomerInfo({
            name: data.customer.name,
            phone: data.customer.phone,
            email: data.customer.email,
          });
          // 自动填充表单
          setFormData(prev => ({
            ...prev,
            contactName: data.customer.name,
            contactPhone: data.customer.phone,
            contactEmail: data.customer.email || '',
          }));
        }
      } catch {
        // 未登录，忽略
      }
    };
    checkAuth();
  }, []);

  const services = [
    { id: 'road-transport', icon: Truck, titleKey: 'services.railway', titleZh: '中俄公路运输', titleRu: 'Автоперевозки Китай-Россия', descKey: 'services.railwayDesc', descZh: '专业公路跨境运输', descRu: 'Профессиональные автоперевозки' },
    { id: 'car-export', icon: Car, titleKey: 'services.road', titleZh: '整车自驾出口', titleRu: 'Экспорт автомобилей', descKey: 'services.roadDesc', descZh: '便捷整车出口服务', descRu: 'Удобный экспорт авто' },
    { id: 'tir-transport', icon: Route, titleKey: 'services.customs', titleZh: 'Tir运输', titleRu: 'TIR перевозки', descKey: 'services.customsDesc', descZh: '国际公路运输服务', descRu: 'Международные автоперевозки' },
    { id: 'railway-transport', icon: Train, titleKey: 'services.warehouse', titleZh: '铁路境外段', titleRu: 'Ж/Д в России', descKey: 'services.warehouseDesc', descZh: '俄罗斯境内铁路运输', descRu: 'Железнодорожные перевозки по РФ' },
    { id: 'railway-import', icon: Ship, titleKey: 'services.railway', titleZh: '铁路大宗货物进口', titleRu: 'Импорт ж/д', descKey: 'services.railwayDesc', descZh: '大宗货物铁路进口', descRu: 'Импорт грузов по ж/д' },
    { id: 'port-agency', icon: Building2, titleKey: 'services.road', titleZh: '满洲里口岸代理', titleRu: 'Агент в Маньчжурии', descKey: 'services.roadDesc', descZh: '口岸通关一站式服务', descRu: 'Таможенное оформление' },
    { id: 'customs-clearance', icon: CheckCircle2, titleKey: 'services.customs', titleZh: '俄罗斯代理清关', titleRu: 'Таможенный брокер', descKey: 'services.customsDesc', descZh: '专业清关代理服务', descRu: 'Профессиональная растаможка' },
    { id: 'oversize-transport', icon: Package, titleKey: 'services.warehouse', titleZh: '超大件设备运输', titleRu: 'Негабаритные грузы', descKey: 'services.warehouseDesc', descZh: '专业超大件运输', descRu: 'Перевозка негабарита' },
    { id: 'project-logistics', icon: Wrench, titleKey: 'services.railway', titleZh: '工程项目物流', titleRu: 'Проектная логистика', descKey: 'services.railwayDesc', descZh: '大型工程项目运输', descRu: 'Логистика для проектов' },
    { id: 'exhibition-logistics', icon: Sparkles, titleKey: 'services.road', titleZh: '展会物流', titleRu: 'Выставочная логистика', descKey: 'services.roadDesc', descZh: '展会货物运输', descRu: 'Перевозка выставок' },
    { id: 'return-service', icon: ArrowLeftRight, titleKey: 'services.customs', titleZh: '退运服务', titleRu: 'Возврат грузов', descKey: 'services.customsDesc', descZh: '货物退运处理', descRu: 'Обработка возвратов' },
    { id: 'ata-carnet', icon: FileText, titleKey: 'services.warehouse', titleZh: 'ATA单证册', titleRu: 'ATA карнет', descKey: 'services.warehouseDesc', descZh: 'ATA专业服务', descRu: 'Услуги ATA' },
    { id: 'repair-service', icon: Wrench, titleKey: 'services.railway', titleZh: '进境修理调试', titleRu: 'Ремонт оборудования', descKey: 'services.railwayDesc', descZh: '设备进境维修', descRu: 'Ввоз для ремонта' },
    { id: 'leasing-trade', icon: Globe, titleKey: 'services.road', titleZh: '中俄租赁贸易', titleRu: 'Лизинг Китай-Россия', descKey: 'services.roadDesc', descZh: '租赁贸易物流', descRu: 'Логистика лизинга' },
  ];

  const advantages = [
    {
      icon: Users,
      titleZh: '团队够"懂行"',
      titleRu: 'Команда экспертов',
      descZh: '由深耕中俄物流资深从业者组成核心团队，熟悉俄语、精通两国海关政策与运输规则，能快速响应客户个性化需求。',
      descRu: 'Наша команда состоит из экспертов с многолетним опытом работы в логистике Китай-Россия, владеющих русским языком и знающих таможенное законодательство обеих стран.',
      featuresZh: ['精通俄语', '熟悉政策', '快速响应'],
      featuresRu: ['Владеем русским', 'Знаем законы', 'Быстро реагируем'],
    },
    {
      icon: Globe,
      titleZh: '网络够"健全"',
      titleRu: 'Широкая сеть',
      descZh: '在满洲里口岸设有专属操作中心，联动俄罗斯境内主要城市合作网点，形成闭环服务网络。',
      descRu: 'Мы имеем операционный центр в Маньчжурии и партнерскую сеть в основных городах России.',
      featuresZh: ['口岸中心', '俄境网点', '闭环网络'],
      featuresRu: ['Центр в порту', 'Сеть в РФ', 'Замкнутый цикл'],
    },
    {
      icon: Package,
      titleZh: '服务够"全面"',
      titleRu: 'Полный сервис',
      descZh: '从常规中俄公路运输到超大件设备、工程项目物流等特殊需求，均能提供定制化解决方案。',
      descRu: 'От стандартных автоперевозок до негабаритных грузов и проектной логистики — мы предлагаем индивидуальные решения.',
      featuresZh: ['定制方案', '一站式服务', '全程无忧'],
      featuresRu: ['Индивидуально', 'Всё включено', 'Без забот'],
    },
  ];

  // 统计数据 - 支持数据库内容
  const stats = useMemo(() => [
    { 
      value: parseInt(getContent('stats_years_value', '10', '10')), 
      suffix: '+', 
      label: getContent('stats_years_label', '行业深耕年限', 'лет опыта'), 
      icon: Award 
    },
    { 
      value: parseInt(getContent('stats_clients_value', '3000', '3000')), 
      suffix: '+', 
      label: getContent('stats_clients_label', '服务企业数量', 'клиентов'), 
      icon: Building2 
    },
    { 
      value: parseInt(getContent('stats_satisfaction_value', '99', '99')), 
      suffix: '%', 
      label: getContent('stats_satisfaction_label', '客户满意度', 'удовлетворенность'), 
      icon: Target 
    },
    { 
      value: parseInt(getContent('stats_service_value', '24', '24')), 
      suffix: 'h', 
      label: getContent('stats_service_label', '全天候服务', 'круглосуточно'), 
      icon: Clock 
    },
  ], [getContent]);

  const features = [
    { icon: Shield, titleZh: '安全可靠', titleRu: 'Безопасность', descZh: '全程货物保险，风险可控', descRu: 'Страхование грузов' },
    { icon: Clock, titleZh: '时效保障', titleRu: 'Сроки', descZh: '准时送达，延误赔付', descRu: 'Гарантия доставки' },
    { icon: MessageSquare, titleZh: '实时追踪', titleRu: 'Отслеживание', descZh: 'GPS定位，随时查询', descRu: 'GPS-мониторинг' },
    { icon: CheckCircle2, titleZh: '清关专业', titleRu: 'Таможня', descZh: '专业团队，快速通关', descRu: 'Быстрое оформление' },
  ];

  // 处理表单输入
  const handleInputChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>
  ) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  // 提交订单
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setSubmitStatus('idle');
    setErrorMessage('');

    try {
      const response = await fetch('/api/orders', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(formData),
      });

      const result = await response.json();

      if (response.ok && result.success) {
        setSubmitStatus('success');
        setFormData({
          contactName: '',
          contactPhone: '',
          contactEmail: '',
          originCity: '',
          destinationCity: '',
          customsCity: '',
          notes: '',
        });
      } else {
        setSubmitStatus('error');
        setErrorMessage(result.error || (locale === 'zh' ? '提交失败，请稍后重试' : 'Ошибка отправки'));
      }
    } catch {
      setSubmitStatus('error');
      setErrorMessage(locale === 'zh' ? '网络错误，请稍后重试' : 'Ошибка сети');
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="min-h-screen bg-white">
      {/* 导航栏 */}
      <nav className="fixed top-0 left-0 right-0 z-50 bg-white/80 backdrop-blur-lg border-b border-gray-100">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center gap-3">
              <div className="relative">
                <div className="w-11 h-11 bg-white rounded-xl flex items-center justify-center shadow-lg border border-gray-100">
                  <img src="/images/logo.png" alt={t.common.logo} className="w-9 h-9 object-contain" />
                </div>
                <div className="absolute -top-1 -right-1 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></div>
              </div>
              <div>
                <h1 className="text-lg font-bold bg-gradient-to-r from-blue-600 to-indigo-700 bg-clip-text text-transparent">{t.common.logo}</h1>
                <p className="text-xs text-gray-400 font-medium tracking-wide">{t.common.logoSub}</p>
              </div>
            </div>
            <div className="hidden md:flex items-center gap-1">
              <a href="#about" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.about}</a>
              <Link href="/services" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.services}</Link>
              <Link href="/cases" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.cases}</Link>
              <Link href="/faqs" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.faqs}</Link>
              <a href="#order" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.order}</a>
              <a href="#contact" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.contact}</a>
            </div>
            <div className="flex items-center gap-2">
              <LanguageSwitcherSimple />
              <Link 
                href="/user/orders" 
                className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all text-sm"
              >
                <User className="w-4 h-4" />
                <span className="hidden sm:inline">{t.nav.userCenter}</span>
              </Link>
              <a 
                href="#order" 
                className="flex items-center gap-2 px-5 py-2.5 bg-gradient-to-r from-blue-600 to-indigo-600 text-white text-sm font-medium rounded-xl hover:shadow-lg hover:shadow-blue-500/25 hover:-translate-y-0.5 transition-all"
              >
                {t.home.inquiryNow}
                <ArrowRight className="w-4 h-4" />
              </a>
            </div>
          </div>
        </div>
      </nav>

      {/* Hero 区域 */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-16">
        {/* 背景图片 */}
        <div className="absolute inset-0">
          <img 
            src="/images/hero-bg.jpeg" 
            alt={locale === 'zh' ? '中俄跨境物流' : 'Логистика Китай-Россия'} 
            className="w-full h-full object-cover"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-slate-900/90 via-blue-900/85 to-indigo-950/90"></div>
        </div>
        <div className="absolute top-0 left-1/4 w-[600px] h-[600px] bg-blue-500/20 rounded-full blur-[120px] animate-pulse"></div>
        <div className="absolute bottom-0 right-1/4 w-[500px] h-[500px] bg-indigo-500/15 rounded-full blur-[100px]"></div>
        <div className="absolute top-1/2 left-0 w-[400px] h-[400px] bg-cyan-500/10 rounded-full blur-[80px]"></div>
        <div className="absolute inset-0 opacity-20" style={{
          backgroundImage: `radial-gradient(circle at 1px 1px, rgba(255,255,255,0.15) 1px, transparent 0)`,
          backgroundSize: '40px 40px'
        }}></div>

        <div className="relative z-10 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <div className="inline-flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-blue-500/20 to-indigo-500/20 backdrop-blur-sm border border-white/10 rounded-full text-white/90 text-sm mb-8">
            <div className="w-2 h-2 bg-green-400 rounded-full animate-pulse"></div>
            <Globe className="w-4 h-4" />
            <span>{t.common.slogan}</span>
          </div>
          
          <h1 className="text-4xl sm:text-5xl lg:text-6xl xl:text-7xl font-bold text-white mb-8 leading-tight">
            <span className="block mb-2">{getContent('hero_title', t.home.heroTitle, t.home.heroTitle)}</span>
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-cyan-300 via-blue-300 to-purple-300">
              {getContent('hero_subtitle', t.home.heroSubtitle, t.home.heroSubtitle)}
            </span>
          </h1>
          
          <p className="text-lg sm:text-xl text-blue-100/80 max-w-2xl mx-auto mb-12 leading-relaxed">
            {getContent('hero_desc', t.home.heroDesc, t.home.heroDesc)}
          </p>

          <div className="flex flex-col sm:flex-row gap-4 justify-center items-center mb-16">
            <a 
              href="#order" 
              className="group inline-flex items-center gap-3 px-8 py-4 bg-white text-blue-700 font-semibold rounded-2xl hover:shadow-2xl hover:shadow-white/20 transform hover:-translate-y-1 transition-all duration-300"
            >
              {t.home.inquiryNow}
              <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
            </a>
            <Link 
              href="/user/orders" 
              className="group inline-flex items-center gap-3 px-8 py-4 bg-white/5 border border-white/20 text-white font-semibold rounded-2xl hover:bg-white/10 backdrop-blur-sm transition-all duration-300"
            >
              <ClipboardList className="w-5 h-5" />
              {locale === 'zh' ? '查询订单' : 'Мои заказы'}
            </Link>
          </div>

          <div className="flex flex-wrap justify-center gap-3 mb-16">
            {features.map((feature, index) => (
              <div key={index} className="flex items-center gap-2 px-4 py-2 bg-white/5 backdrop-blur-sm border border-white/10 rounded-xl text-white/80 text-sm">
                <feature.icon className="w-4 h-4 text-cyan-400" />
                <span>{locale === 'zh' ? feature.titleZh : feature.titleRu}</span>
              </div>
            ))}
          </div>
        </div>

        <div className="absolute bottom-0 left-0 right-0 h-32 bg-gradient-to-t from-white to-transparent"></div>
        <div className="absolute bottom-8 left-1/2 -translate-x-1/2 flex flex-col items-center gap-2 text-white/50">
          <span className="text-xs">{locale === 'zh' ? '向下滚动' : 'Прокрутите вниз'}</span>
          <ChevronDown className="w-5 h-5 animate-bounce" />
        </div>
      </section>

      {/* 数据统计 */}
      <section className="relative -mt-16 z-20 py-8">
        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="bg-white rounded-3xl shadow-xl shadow-gray-200/50 border border-gray-100 p-8">
            <div className="grid grid-cols-2 lg:grid-cols-4 gap-8">
              {stats.map((stat, index) => (
                <div key={index} className="text-center group">
                  <div className="inline-flex items-center justify-center w-12 h-12 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl mb-4 group-hover:scale-110 transition-transform">
                    <stat.icon className="w-6 h-6 text-blue-600" />
                  </div>
                  <div className="text-3xl sm:text-4xl font-bold bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent mb-1">
                    <AnimatedNumber end={stat.value} suffix={stat.suffix} />
                  </div>
                  <div className="text-sm text-gray-500">{stat.label}</div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* 关于我们 */}
      <section id="about" className="py-24 bg-gradient-to-b from-white to-gray-50 overflow-hidden">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-2 gap-16 items-center">
            <div className="relative">
              <div className="absolute -top-10 -left-10 w-72 h-72 bg-blue-100 rounded-full blur-3xl opacity-50"></div>
              <div className="absolute -bottom-10 -right-10 w-72 h-72 bg-indigo-100 rounded-full blur-3xl opacity-50"></div>
              <div className="relative grid grid-cols-2 gap-4">
                <div className="space-y-4">
                  <div className="relative overflow-hidden rounded-2xl shadow-xl aspect-[4/3]">
                    <img 
                      src="/images/warehouse.jpeg" 
                      alt={locale === 'zh' ? '物流仓库' : 'Склад'} 
                      className="absolute inset-0 w-full h-full object-cover hover:scale-105 transition-transform duration-500"
                    />
                    <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/60 to-transparent p-3">
                      <span className="text-white text-sm font-medium">{locale === 'zh' ? '现代化仓储设施' : 'Современные склады'}</span>
                    </div>
                  </div>
                  <div className="relative overflow-hidden rounded-2xl shadow-xl aspect-[4/3]">
                    <img 
                      src="/images/railway.jpeg" 
                      alt={locale === 'zh' ? '铁路运输' : 'Ж/Д перевозки'} 
                      className="absolute inset-0 w-full h-full object-cover hover:scale-105 transition-transform duration-500"
                    />
                    <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/60 to-transparent p-3">
                      <span className="text-white text-sm font-medium">{locale === 'zh' ? '铁路货运网络' : 'Ж/Д сеть'}</span>
                    </div>
                  </div>
                </div>
                <div className="space-y-4 pt-8">
                  <div className="relative overflow-hidden rounded-2xl shadow-xl aspect-[4/3]">
                    <img 
                      src="/images/team.jpeg" 
                      alt={locale === 'zh' ? '专业团队' : 'Команда'} 
                      className="absolute inset-0 w-full h-full object-cover hover:scale-105 transition-transform duration-500"
                    />
                    <div className="absolute bottom-0 left-0 right-0 bg-gradient-to-t from-black/60 to-transparent p-3">
                      <span className="text-white text-sm font-medium">{locale === 'zh' ? '专业服务团队' : 'Профессиональная команда'}</span>
                    </div>
                  </div>
                  <div className="relative bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 rounded-2xl p-6 shadow-xl aspect-[4/3] flex flex-col justify-center">
                    <div className="absolute top-0 right-0 w-20 h-20 bg-white/10 rounded-full blur-xl"></div>
                    <div className="relative z-10">
                      <div className="text-4xl font-bold text-white mb-1">10+</div>
                      <div className="text-blue-200 text-sm">{locale === 'zh' ? '年行业深耕' : 'лет опыта'}</div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            <div>
              <div className="inline-flex items-center gap-2 px-3 py-1 bg-blue-50 text-blue-600 text-sm font-medium rounded-full mb-6">
                <Building2 className="w-4 h-4" />
                {t.nav.about}
              </div>
              <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold text-gray-900 mb-6 leading-tight">
                {getContent('about_title', 
                  locale === 'zh' ? '深耕中俄货运领域' : 'Эксперты логистики',
                  locale === 'zh' ? '深耕中俄货运领域' : 'Эксперты логистики'
                )}
                <br />
                <span className="text-transparent bg-clip-text bg-gradient-to-r from-blue-600 to-indigo-600">
                  {getContent('about_subtitle', 
                    locale === 'zh' ? '成为可靠的跨境桥梁' : 'Китай — Россия',
                    locale === 'zh' ? '成为可靠的跨境桥梁' : 'Китай — Россия'
                  )}
                </span>
              </h2>
              <p className="text-lg text-gray-600 leading-relaxed mb-6">
                {getContent('about_desc1', 
                  '满洲里欣琪国际货运代理有限责任公司深耕中俄货运领域多年，凭借扎根满洲里口岸的地理优势与对中俄贸易规则的深刻理解，成为连接两国货物往来的可靠桥梁。',
                  'Маньчжурия Синьци Международное Экспедирование — эксперт в сфере логистики Китай-Россия с многолетним опытом работы на границе.'
                )}
              </p>
              <p className="text-lg text-gray-600 leading-relaxed mb-8">
                {getContent('about_desc2',
                  '作为中俄贸易的见证者与参与者，欣琪国际始终以"安全、高效、透明"为服务准则，累计为数千家企业完成跨境货物运输。',
                  'Мы обслужили тысячи компаний, обеспечивая безопасность, эффективность и прозрачность на каждом этапе.'
                )}
              </p>
              <div className="grid grid-cols-3 gap-4">
                {[
                  { icon: Shield, labelZh: '安全可靠', labelRu: 'Безопасность' },
                  { icon: Clock, labelZh: '高效快捷', labelRu: 'Скорость' },
                  { icon: CheckCircle2, labelZh: '全程透明', labelRu: 'Прозрачность' },
                ].map((item, index) => (
                  <div key={index} className="flex items-center gap-2 p-3 bg-white rounded-xl border border-gray-100 shadow-sm">
                    <div className="w-8 h-8 bg-blue-50 rounded-lg flex items-center justify-center">
                      <item.icon className="w-4 h-4 text-blue-600" />
                    </div>
                    <span className="text-sm font-medium text-gray-700">{locale === 'zh' ? item.labelZh : item.labelRu}</span>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* 服务内容 */}
      <section id="services" className="py-24 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-blue-50 text-blue-600 text-sm font-medium rounded-full mb-6">
              <Package className="w-4 h-4" />
              {t.nav.services}
            </div>
            <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">
              {getContent('services_title', 
                locale === 'zh' ? '全方位中俄货运解决方案' : 'Полный спектр услуг',
                locale === 'zh' ? '全方位中俄货运解决方案' : 'Полный спектр услуг'
              )}
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              {getContent('services_desc',
                locale === 'zh' ? '满足不同客户的多样化需求，提供一站式跨境物流服务' : 'Индивидуальные решения для каждого клиента',
                locale === 'zh' ? '满足不同客户的多样化需求，提供一站式跨境物流服务' : 'Индивидуальные решения для каждого клиента'
              )}
            </p>
          </div>
          <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
            {services.map((service, index) => (
              <Link key={index} href={`/services/${service.id}`} className="group relative bg-white rounded-2xl p-5 border border-gray-100 hover:border-blue-300 hover:shadow-lg hover:shadow-blue-500/10 transition-all duration-300 cursor-pointer overflow-hidden block">
                <div className="absolute inset-0 bg-gradient-to-br from-blue-500/0 to-indigo-500/0 group-hover:from-blue-500/5 group-hover:to-indigo-500/5 transition-all"></div>
                <div className="relative z-10">
                  <div className="w-10 h-10 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl flex items-center justify-center mb-3 group-hover:from-blue-100 group-hover:to-indigo-100 transition-colors">
                    <service.icon className="w-5 h-5 text-blue-600" />
                  </div>
                  <h3 className="font-semibold text-gray-900 mb-1 group-hover:text-blue-600 transition-colors">
                    {locale === 'zh' ? service.titleZh : service.titleRu}
                  </h3>
                  <p className="text-xs text-gray-400">{locale === 'zh' ? service.descZh : service.descRu}</p>
                </div>
              </Link>
            ))}
          </div>
        </div>
      </section>

      {/* 在线询价 */}
      <section id="order" className="py-24 bg-gradient-to-b from-gray-50 to-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-blue-50 text-blue-600 text-sm font-medium rounded-full mb-6">
              <Send className="w-4 h-4" />
              {t.home.order.title}
            </div>
            <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">
              {getContent('order_title',
                locale === 'zh' ? '立即提交您的运输需求' : 'Отправьте заявку сейчас',
                locale === 'zh' ? '立即提交您的运输需求' : 'Отправьте заявку сейчас'
              )}
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              {t.home.order.subtitle}
            </p>
          </div>

          <div className="grid lg:grid-cols-5 gap-8">
            {/* 优势说明 */}
            <div className="lg:col-span-2">
              <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm sticky top-24">
                <h3 className="text-lg font-semibold text-gray-900 mb-6 flex items-center gap-2">
                  <Award className="w-5 h-5 text-blue-600" />
                  {locale === 'zh' ? '为什么选择我们' : 'Почему выбирают нас'}
                </h3>
                <div className="space-y-4">
                  <div className="flex items-start gap-4 p-3 bg-gray-50 rounded-xl">
                    <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center flex-shrink-0">
                      <Clock className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <div className="font-medium text-gray-900">{locale === 'zh' ? '快速响应' : 'Быстрый ответ'}</div>
                      <div className="text-sm text-gray-500">{locale === 'zh' ? '24小时内专人联系' : 'Ответим за 24 часа'}</div>
                    </div>
                  </div>
                  <div className="flex items-start gap-4 p-3 bg-gray-50 rounded-xl">
                    <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center flex-shrink-0">
                      <Shield className="w-5 h-5 text-green-600" />
                    </div>
                    <div>
                      <div className="font-medium text-gray-900">{locale === 'zh' ? '安全可靠' : 'Надежность'}</div>
                      <div className="text-sm text-gray-500">{locale === 'zh' ? '全程货物保险' : 'Страхование грузов'}</div>
                    </div>
                  </div>
                  <div className="flex items-start gap-4 p-3 bg-gray-50 rounded-xl">
                    <div className="w-10 h-10 bg-purple-100 rounded-lg flex items-center justify-center flex-shrink-0">
                      <Route className="w-5 h-5 text-purple-600" />
                    </div>
                    <div>
                      <div className="font-medium text-gray-900">{locale === 'zh' ? '专业路线' : 'Экспертные маршруты'}</div>
                      <div className="text-sm text-gray-500">{locale === 'zh' ? '深耕中俄物流多年' : 'Многолетний опыт'}</div>
                    </div>
                  </div>
                  <div className="flex items-start gap-4 p-3 bg-gray-50 rounded-xl">
                    <div className="w-10 h-10 bg-amber-100 rounded-lg flex items-center justify-center flex-shrink-0">
                      <MessageSquare className="w-5 h-5 text-amber-600" />
                    </div>
                    <div>
                      <div className="font-medium text-gray-900">{locale === 'zh' ? '实时沟通' : 'Связь 24/7'}</div>
                      <div className="text-sm text-gray-500">{locale === 'zh' ? '在线客服实时沟通' : 'Онлайн поддержка'}</div>
                    </div>
                  </div>
                </div>
                <div className="mt-6 pt-6 border-t border-gray-100">
                  <Link
                    href="/user/orders"
                    className="flex items-center justify-center gap-2 w-full px-4 py-3 bg-blue-50 text-blue-600 font-medium rounded-xl hover:bg-blue-100 transition-colors"
                  >
                    <User className="w-4 h-4" />
                    {locale === 'zh' ? '查询我的订单' : 'Мои заказы'}
                    <ArrowRight className="w-4 h-4" />
                  </Link>
                </div>
              </div>
            </div>

            {/* 订单表单 */}
            <div className="lg:col-span-3">
              {submitStatus === 'success' ? (
                <div className="bg-gradient-to-br from-green-50 to-emerald-50 rounded-3xl p-8 border border-green-100">
                  <div className="text-center">
                    <div className="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
                      <CheckCircle2 className="w-10 h-10 text-green-600" />
                    </div>
                    <h3 className="text-2xl font-bold text-gray-900 mb-3">{t.home.order.submitSuccess}</h3>
                    <p className="text-gray-600 mb-6">
                      {t.home.order.submitSuccessDesc}
                    </p>
                    <button
                      onClick={() => setSubmitStatus('idle')}
                      className="inline-flex items-center gap-2 px-6 py-3 bg-green-600 text-white font-medium rounded-xl hover:bg-green-700 transition-colors"
                    >
                      {locale === 'zh' ? '继续询价' : 'Новая заявка'}
                      <ArrowRight className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              ) : (
                <form onSubmit={handleSubmit} className="space-y-6">
                  {/* 联系人信息 */}
                  <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
                    <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <User className="w-5 h-5 text-blue-600" />
                      {t.home.order.contactName}
                    </h4>
                    {isLoggedIn && customerInfo ? (
                      /* 已登录用户显示已填充的信息 */
                      <div className="bg-blue-50 rounded-xl p-4">
                        <div className="flex items-center justify-between">
                          <div className="space-y-2">
                            <div className="flex items-center gap-2 text-gray-700">
                              <User className="w-4 h-4 text-gray-400" />
                              <span className="font-medium">{customerInfo.name}</span>
                            </div>
                            <div className="flex items-center gap-2 text-gray-700">
                              <Phone className="w-4 h-4 text-gray-400" />
                              <span>{customerInfo.phone}</span>
                            </div>
                            {customerInfo.email && (
                              <div className="flex items-center gap-2 text-gray-700">
                                <Mail className="w-4 h-4 text-gray-400" />
                                <span>{customerInfo.email}</span>
                              </div>
                            )}
                          </div>
                          <div className="text-xs text-blue-600 bg-blue-100 px-2 py-1 rounded">
                            {locale === 'zh' ? '已登录' : 'Вход выполнен'}
                          </div>
                        </div>
                      </div>
                    ) : (
                      /* 未登录用户显示输入框 */
                      <>
                        <div className="grid sm:grid-cols-2 gap-4">
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              {locale === 'zh' ? '姓名' : 'Имя'} <span className="text-red-500">*</span>
                            </label>
                            <input type="text" name="contactName" value={formData.contactName} onChange={handleInputChange} required placeholder={t.home.order.contactNamePlaceholder}
                              className="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all" />
                          </div>
                          <div>
                            <label className="block text-sm font-medium text-gray-700 mb-2">
                              {locale === 'zh' ? '手机号' : 'Телефон'} <span className="text-red-500">*</span>
                            </label>
                            <div className="relative">
                              <Phone className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                              <input type="tel" name="contactPhone" value={formData.contactPhone} onChange={handleInputChange} required placeholder={t.home.order.contactPhonePlaceholder}
                                className="w-full pl-11 pr-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all" />
                            </div>
                          </div>
                        </div>
                        <div className="mt-4">
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            {t.home.order.contactEmail} ({locale === 'zh' ? '选填' : 'опционально'})
                          </label>
                          <div className="relative">
                            <Mail className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                            <input type="email" name="contactEmail" value={formData.contactEmail} onChange={handleInputChange} placeholder={t.home.order.contactEmailPlaceholder}
                              className="w-full pl-11 pr-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all" />
                          </div>
                        </div>
                        <div className="mt-4 pt-4 border-t border-gray-100">
                          <p className="text-sm text-gray-500">
                            {t.user.noAccount}
                            <Link href="/auth" className="text-blue-600 hover:underline ml-1">{t.user.login}</Link>
                          </p>
                        </div>
                      </>
                    )}
                  </div>

                  {/* 运输信息 */}
                  <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
                    <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <MapPin className="w-5 h-5 text-blue-600" />
                      {locale === 'zh' ? '运输信息' : 'Информация о грузе'}
                    </h4>
                    <div className="space-y-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          {t.home.order.originCity} <span className="text-red-500">*</span>
                        </label>
                        <div className="relative">
                          <MapPin className="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-blue-500" />
                          <input 
                            type="text" 
                            name="originCity" 
                            value={formData.originCity} 
                            onChange={handleInputChange} 
                            required 
                            placeholder={t.home.order.originCityPlaceholder}
                            className="w-full pl-11 pr-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                          />
                        </div>
                      </div>
                      <div className="grid sm:grid-cols-2 gap-4">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            {t.home.order.destinationCity} <span className="text-red-500">*</span>
                          </label>
                          <input 
                            type="text" 
                            name="destinationCity" 
                            value={formData.destinationCity} 
                            onChange={handleInputChange} 
                            required 
                            placeholder={t.home.order.destinationCityPlaceholder}
                            className="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                          />
                        </div>
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-2">
                            {t.home.order.customsCity} <span className="text-red-500">*</span>
                          </label>
                          <input 
                            type="text" 
                            name="customsCity" 
                            value={formData.customsCity} 
                            onChange={handleInputChange} 
                            required 
                            placeholder={t.home.order.customsCityPlaceholder}
                            className="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all"
                          />
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* 备注信息 */}
                  <div className="bg-white rounded-2xl p-6 border border-gray-100 shadow-sm">
                    <h4 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                      <FileText className="w-5 h-5 text-blue-600" />
                      {t.home.order.notes}
                    </h4>
                    <textarea name="notes" value={formData.notes} onChange={handleInputChange} rows={4} placeholder={t.home.order.notesPlaceholder}
                      className="w-full px-4 py-3 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all resize-none" />
                  </div>

                  {submitStatus === 'error' && (
                    <div className="bg-red-50 border border-red-200 rounded-xl p-4 text-red-600 text-sm">{errorMessage}</div>
                  )}

                  <button type="submit" disabled={isSubmitting}
                    className="w-full flex items-center justify-center gap-3 px-8 py-4 bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-semibold rounded-2xl hover:shadow-lg hover:shadow-blue-500/25 disabled:opacity-50 disabled:cursor-not-allowed transition-all">
                    {isSubmitting ? (
                      <>
                        <Loader2 className="w-5 h-5 animate-spin" />
                        {t.home.order.submitting}
                      </>
                    ) : (
                      <>
                        <Send className="w-5 h-5" />
                        {t.home.order.submit}
                      </>
                    )}
                  </button>
                </form>
              )}
            </div>
          </div>
        </div>
      </section>

      {/* 核心竞争力 */}
      <section id="advantages" className="py-24 bg-gradient-to-b from-white to-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-blue-50 text-blue-600 text-sm font-medium rounded-full mb-6">
              <Award className="w-4 h-4" />
              {locale === 'zh' ? '核心竞争力' : 'Наши преимущества'}
            </div>
            <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">
              {getContent('advantages_title',
                locale === 'zh' ? '三重专业保障' : 'Три уровня гарантии',
                locale === 'zh' ? '三重专业保障' : 'Три уровня гарантии'
              )}
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              {getContent('advantages_desc',
                locale === 'zh' ? '让我们成为中俄货运领域的可靠合作伙伴' : 'Почему нам доверяют',
                locale === 'zh' ? '让我们成为中俄货运领域的可靠合作伙伴' : 'Почему нам доверяют'
              )}
            </p>
          </div>
          <div className="grid lg:grid-cols-3 gap-8">
            {advantages.map((item, index) => (
              <div key={index} className="group relative bg-white rounded-3xl p-8 border border-gray-100 hover:border-blue-200 hover:shadow-2xl hover:shadow-blue-500/10 transition-all duration-500">
                <div className="absolute top-0 right-0 w-40 h-40 bg-gradient-to-br from-blue-50 to-transparent rounded-bl-3xl opacity-0 group-hover:opacity-100 transition-opacity duration-500"></div>
                <div className="relative z-10">
                  <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform shadow-lg shadow-blue-500/25">
                    <item.icon className="w-8 h-8 text-white" />
                  </div>
                  <h3 className="text-xl font-bold text-gray-900 mb-4">{locale === 'zh' ? item.titleZh : item.titleRu}</h3>
                  <p className="text-gray-600 leading-relaxed mb-6">{locale === 'zh' ? item.descZh : item.descRu}</p>
                  <div className="flex flex-wrap gap-2">
                    {(locale === 'zh' ? item.featuresZh : item.featuresRu).map((feature, idx) => (
                      <span key={idx} className="px-3 py-1 bg-blue-50 text-blue-600 text-xs font-medium rounded-full">{feature}</span>
                    ))}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* 成功案例 */}
      <section className="py-24 bg-white overflow-hidden">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-blue-50 text-blue-600 text-sm font-medium rounded-full mb-6">
              <Globe className="w-4 h-4" />
              {locale === 'zh' ? '成功案例' : 'Успешные проекты'}
            </div>
            <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">
              {getContent('cases_title',
                locale === 'zh' ? '值得信赖的合作伙伴' : 'Надежный партнер',
                locale === 'zh' ? '值得信赖的合作伙伴' : 'Надежный партнер'
              )}
            </h2>
            <p className="text-lg text-gray-600 max-w-2xl mx-auto">
              {getContent('cases_desc',
                locale === 'zh' ? '我们已为数千家企业提供专业的中俄物流服务' : 'Мы обслужили тысячи клиентов',
                locale === 'zh' ? '我们已为数千家企业提供专业的中俄物流服务' : 'Мы обслужили тысячи клиентов'
              )}
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-6">
            {/* 案例卡片 */}
            <div className="group relative bg-gradient-to-br from-gray-50 to-white rounded-3xl overflow-hidden border border-gray-100 hover:shadow-2xl hover:shadow-blue-500/10 transition-all duration-500">
              <div className="relative h-52 overflow-hidden bg-gray-100">
                <img 
                  src="/images/car-export.jpeg"
                  alt={locale === 'zh' ? '汽车出口运输' : 'Экспорт автомобилей'} 
                  className="absolute inset-0 w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
                <div className="absolute bottom-4 left-4 right-4">
                  <span className="inline-block px-3 py-1 bg-blue-600 text-white text-xs font-medium rounded-full mb-2">{locale === 'zh' ? '公路运输' : 'Авто'}</span>
                  <h3 className="text-white font-semibold text-lg">{locale === 'zh' ? '整车出口俄罗斯' : 'Экспорт авто в РФ'}</h3>
                </div>
              </div>
              <div className="p-6">
                <p className="text-gray-600 text-sm mb-4">
                  {locale === 'zh' 
                    ? '为中国某汽车品牌提供整车出口俄罗斯服务，年运输量超过2000台，涵盖EAC认证、清关等全流程服务。'
                    : 'Ежегодно перевозим более 2000 автомобилей для китайских брендов, включая EAC-сертификацию и таможенное оформление.'
                  }
                </p>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-400">{locale === 'zh' ? '年运输量' : 'В год'}</span>
                  <span className="text-blue-600 font-semibold">2000+ {locale === 'zh' ? '台' : 'авто'}</span>
                </div>
              </div>
            </div>

            <div className="group relative bg-gradient-to-br from-gray-50 to-white rounded-3xl overflow-hidden border border-gray-100 hover:shadow-2xl hover:shadow-blue-500/10 transition-all duration-500">
              <div className="relative h-52 overflow-hidden bg-gray-100">
                <img 
                  src="/images/railway.jpeg"
                  alt={locale === 'zh' ? '大宗货物进口' : 'Импорт грузов'} 
                  className="absolute inset-0 w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
                <div className="absolute bottom-4 left-4 right-4">
                  <span className="inline-block px-3 py-1 bg-green-600 text-white text-xs font-medium rounded-full mb-2">{locale === 'zh' ? '铁路运输' : 'Ж/Д'}</span>
                  <h3 className="text-white font-semibold text-lg">{locale === 'zh' ? '大宗货物进口' : 'Импорт леса'}</h3>
                </div>
              </div>
              <div className="p-6">
                <p className="text-gray-600 text-sm mb-4">
                  {locale === 'zh'
                    ? '为国内木材加工企业进口俄罗斯原木，通过铁路运输年承运量超过10万立方米，稳定高效。'
                    : 'Ежегодно импортируем более 100 000 м³ древесины для китайских предприятий.'
                  }
                </p>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-400">{locale === 'zh' ? '年承运量' : 'В год'}</span>
                  <span className="text-green-600 font-semibold">100 000+ m³</span>
                </div>
              </div>
            </div>

            <div className="group relative bg-gradient-to-br from-gray-50 to-white rounded-3xl overflow-hidden border border-gray-100 hover:shadow-2xl hover:shadow-blue-500/10 transition-all duration-500">
              <div className="relative h-52 overflow-hidden bg-gray-100">
                <img 
                  src="/images/warehouse.jpeg"
                  alt={locale === 'zh' ? '工程项目物流' : 'Проектная логистика'} 
                  className="absolute inset-0 w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
                <div className="absolute bottom-4 left-4 right-4">
                  <span className="inline-block px-3 py-1 bg-purple-600 text-white text-xs font-medium rounded-full mb-2">{locale === 'zh' ? '特殊物流' : 'Спецгрузы'}</span>
                  <h3 className="text-white font-semibold text-lg">{locale === 'zh' ? '工程项目物流' : 'Проектная логистика'}</h3>
                </div>
              </div>
              <div className="p-6">
                <p className="text-gray-600 text-sm mb-4">
                  {locale === 'zh'
                    ? '为某大型能源企业提供俄罗斯项目设备运输服务，包括超大件设备、生产线等，全程跟踪保障。'
                    : 'Перевозим оборудование для энергетических проектов, включая негабаритные грузы.'
                  }
                </p>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-400">{locale === 'zh' ? '项目价值' : 'Стоимость'}</span>
                  <span className="text-purple-600 font-semibold">50 000 000+ ¥</span>
                </div>
              </div>
            </div>
          </div>

          {/* 客户类型 */}
          <div className="mt-16 bg-gradient-to-r from-blue-50 via-indigo-50 to-purple-50 rounded-3xl p-8">
            <h3 className="text-xl font-bold text-gray-900 text-center mb-8">
              {getContent('clients_title',
                locale === 'zh' ? '我们服务的客户类型' : 'Наши клиенты',
                locale === 'zh' ? '我们服务的客户类型' : 'Наши клиенты'
              )}
            </h3>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
              {[
                { icon: Building2, labelZh: '制造业企业', labelRu: 'Производство', count: '500+' },
                { icon: Truck, labelZh: '汽车经销商', labelRu: 'Автодилеры', count: '200+' },
                { icon: Package, labelZh: '跨境电商', labelRu: 'E-commerce', count: '300+' },
                { icon: Globe, labelZh: '外贸公司', labelRu: 'Экспортеры', count: '800+' },
              ].map((item, index) => (
                <div key={index} className="text-center">
                  <div className="inline-flex items-center justify-center w-12 h-12 bg-white rounded-xl shadow-sm mb-3">
                    <item.icon className="w-6 h-6 text-blue-600" />
                  </div>
                  <div className="font-medium text-gray-900">{locale === 'zh' ? item.labelZh : item.labelRu}</div>
                  <div className="text-sm text-blue-600">{item.count}</div>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* 联系我们 */}
      <section id="contact" className="py-24 relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-slate-900 via-blue-900 to-indigo-950">
          <div className="absolute top-0 left-0 w-[500px] h-[500px] bg-blue-500/10 rounded-full blur-[100px]"></div>
          <div className="absolute bottom-0 right-0 w-[600px] h-[600px] bg-indigo-500/10 rounded-full blur-[120px]"></div>
          <div className="absolute inset-0 opacity-20" style={{
            backgroundImage: `radial-gradient(circle at 1px 1px, rgba(255,255,255,0.1) 1px, transparent 0)`,
            backgroundSize: '40px 40px'
          }}></div>
        </div>

        <div className="relative z-10 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-white/10 text-white/80 text-sm font-medium rounded-full mb-6 backdrop-blur-sm">
              <Phone className="w-4 h-4" />
              {t.home.contact.title}
            </div>
            <h2 className="text-3xl sm:text-4xl font-bold text-white mb-4">
              {getContent('contact_title',
                locale === 'zh' ? '期待与您的合作' : 'Ждем вашего звонка',
                locale === 'zh' ? '期待与您的合作' : 'Ждем вашего звонка'
              )}
            </h2>
            <p className="text-lg text-blue-100/80 max-w-2xl mx-auto">
              {t.home.contact.subtitle}
            </p>
          </div>

          <div className="grid lg:grid-cols-3 gap-6">
            <div className="group bg-white/5 backdrop-blur-sm border border-white/10 rounded-3xl p-8 hover:bg-white/10 transition-all duration-300">
              <div className="w-14 h-14 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-2xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                <MapPin className="w-7 h-7 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-white mb-3">{t.home.contact.address}</h3>
              <p className="text-blue-100/80 text-sm leading-relaxed">{t.home.contact.addressValue}</p>
            </div>

            <div className="group bg-white/5 backdrop-blur-sm border border-white/10 rounded-3xl p-8 hover:bg-white/10 transition-all duration-300">
              <div className="w-14 h-14 bg-gradient-to-br from-cyan-500 to-blue-600 rounded-2xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                <Phone className="w-7 h-7 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-white mb-3">{t.home.contact.phone}</h3>
              <div className="space-y-2">
                <a href="tel:13327011775" className="block text-lg text-white hover:text-cyan-300 transition-colors">133 2701 1775</a>
                <a href="tel:15540575587" className="block text-lg text-white hover:text-cyan-300 transition-colors">155 4057 5587</a>
              </div>
            </div>

            <div className="group bg-white/5 backdrop-blur-sm border border-white/10 rounded-3xl p-8 hover:bg-white/10 transition-all duration-300">
              <div className="w-14 h-14 bg-gradient-to-br from-purple-500 to-indigo-600 rounded-2xl flex items-center justify-center mb-6 group-hover:scale-110 transition-transform">
                <Mail className="w-7 h-7 text-white" />
              </div>
              <h3 className="text-xl font-semibold text-white mb-3">{t.home.contact.email}</h3>
              <a href="mailto:mzlxqgj@163.com" className="text-lg text-white hover:text-purple-300 transition-colors">mzlxqgj@163.com</a>
            </div>
          </div>

          <div className="text-center mt-12">
            <a href="tel:13327011775" className="inline-flex items-center gap-3 px-8 py-4 bg-white text-blue-700 font-semibold rounded-2xl hover:shadow-2xl hover:shadow-white/20 transform hover:-translate-y-1 transition-all duration-300">
              <Phone className="w-5 h-5" />
              {locale === 'zh' ? '立即拨打电话咨询' : 'Позвоните сейчас'}
              <ArrowRight className="w-5 h-5" />
            </a>
          </div>
        </div>
      </section>

      {/* 页脚 */}
      <footer className="bg-slate-900 text-gray-400 py-16 border-t border-white/5">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="grid lg:grid-cols-4 gap-12 mb-12">
            <div className="lg:col-span-2">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-11 h-11 bg-white rounded-xl flex items-center justify-center border border-gray-700">
                  <img src="/images/logo.png" alt={t.common.logo} className="w-9 h-9 object-contain" />
                </div>
                <div>
                  <h3 className="text-white font-semibold text-lg">{t.common.logo}</h3>
                  <p className="text-xs text-gray-500">{t.common.logoSub}</p>
                </div>
              </div>
              <p className="text-gray-400 leading-relaxed mb-6 max-w-md">
                {locale === 'zh' 
                  ? '专业的中俄跨境物流解决方案提供商，深耕满洲里口岸多年，为您的货物保驾护航，让跨境运输更简单。'
                  : 'Профессиональный провайдер логистических решений Китай-Россия с многолетним опытом работы на границе.'
                }
              </p>
              <div className="flex gap-4">
                <a href="tel:13327011775" className="w-10 h-10 bg-white/5 hover:bg-white/10 rounded-lg flex items-center justify-center transition-colors">
                  <Phone className="w-5 h-5 text-gray-400" />
                </a>
                <a href="mailto:mzlxqgj@163.com" className="w-10 h-10 bg-white/5 hover:bg-white/10 rounded-lg flex items-center justify-center transition-colors">
                  <Mail className="w-5 h-5 text-gray-400" />
                </a>
              </div>
            </div>
            <div>
              <h4 className="text-white font-semibold mb-4">{locale === 'zh' ? '快速链接' : 'Ссылки'}</h4>
              <ul className="space-y-3">
                <li><a href="#about" className="text-gray-400 hover:text-white transition-colors text-sm">{t.nav.about}</a></li>
                <li><a href="#services" className="text-gray-400 hover:text-white transition-colors text-sm">{t.nav.services}</a></li>
                <li><a href="#order" className="text-gray-400 hover:text-white transition-colors text-sm">{t.nav.order}</a></li>
                <li><Link href="/user/orders" className="text-gray-400 hover:text-white transition-colors text-sm">{t.nav.userCenter}</Link></li>
              </ul>
            </div>
            <div>
              <h4 className="text-white font-semibold mb-4">{locale === 'zh' ? '服务项目' : 'Услуги'}</h4>
              <ul className="space-y-3 text-sm">
                <li><span className="text-gray-400">{t.services.railway}</span></li>
                <li><span className="text-gray-400">{t.services.road}</span></li>
                <li><span className="text-gray-400">{t.services.customs}</span></li>
                <li><span className="text-gray-400">{locale === 'zh' ? '超大件运输' : 'Негабарит'}</span></li>
              </ul>
            </div>
          </div>
          <div className="pt-8 border-t border-white/5 flex flex-col md:flex-row justify-between items-center gap-4">
            <p className="text-sm text-gray-500">{t.home.footer.copyright}</p>
            <p className="text-xs text-gray-600">{t.home.contact.addressValue}</p>
          </div>
        </div>
      </footer>
    </div>
  );
}
